<?php
/**
 * Utility functions
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

/**
 * Log message
 */
function log_message($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] [$level] $message\n";
    file_put_contents(LOG_FILE, $log_entry, FILE_APPEND);
}

/**
 * Validate hardware ID format
 */
function validate_hardware_id_format($hardware_id) {
    if (empty($hardware_id)) {
        return false;
    }
    if (strlen($hardware_id) !== 64) {
        return false;
    }
    return ctype_xdigit($hardware_id); // Check if hex
}

/**
 * Validate email format
 */
function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate license type
 */
function validate_license_type($type) {
    $valid_types = ['basic', 'premium', 'enterprise', 'trial'];
    return in_array(strtolower($type), $valid_types);
}

/**
 * Sanitize input
 */
function sanitize_input($text, $max_length = 1000) {
    if (empty($text)) {
        return '';
    }
    // Remove any characters that are not printable (except newlines, tabs, etc.)
    $text = filter_var($text, FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $text = htmlspecialchars_decode($text, ENT_QUOTES);
    $text = substr($text, 0, $max_length);
    return trim($text);
}

/**
 * Generate license key
 */
function generate_license_key($data) {
    $json = json_encode($data);
    return base64_encode($json);
}

/**
 * Decode license key
 */
function decode_license_key($license_key) {
    try {
        $decoded = base64_decode($license_key, true);
        if ($decoded === false) {
            return null;
        }
        return json_decode($decoded, true);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Validate license
 */
function validate_license($license_key, $hardware_id) {
    // Get license from database
    $license = get_license_by_key($license_key);
    
    if (!$license) {
        return [
            'valid' => false,
            'error' => 'License not found'
        ];
    }
    
    if (!$license['is_active']) {
        return [
            'valid' => false,
            'error' => 'License is deactivated'
        ];
    }
    
    // Check expiration
    $expires = strtotime($license['expires']);
    $now = time();
    if ($expires < $now) {
        return [
            'valid' => false,
            'error' => 'License has expired'
        ];
    }
    
    // Check hardware ID binding
    if (!empty($license['hardware_id'])) {
        if ($license['hardware_id'] !== $hardware_id) {
            return [
                'valid' => false,
                'error' => 'License is bound to a different machine ID'
            ];
        }
    } else {
        // First activation - bind to this hardware ID
        update_license_db($license['id'], ['hardware_id' => $hardware_id]);
        $license['hardware_id'] = $hardware_id;
    }
    
    // Update activated_at if not set
    if (empty($license['activated_at'])) {
        update_license_db($license['id'], ['activated_at' => date('Y-m-d H:i:s')]);
    }
    
    return [
        'valid' => true,
        'license_data' => [
            'license_type' => $license['license_type'],
            'expires' => $license['expires'],
            'max_threads' => $license['max_threads'],
            'max_recipients' => $license['max_recipients'],
            'hardware_id' => $license['hardware_id']
        ],
        'features' => $license['features']
    ];
}

/**
 * Create license
 */
function create_license($data) {
    // Validate input
    $license_type = sanitize_input($data['license_type'] ?? 'basic', 50);
    $hardware_id = sanitize_input($data['hardware_id'] ?? '', 100);
    $customer_name = sanitize_input($data['customer_name'] ?? '', 200);
    $customer_email = sanitize_input($data['customer_email'] ?? '', 200);
    $expires_days = (int)($data['expires_days'] ?? 365);
    $max_threads = (int)($data['max_threads'] ?? 10);
    $max_recipients = (int)($data['max_recipients'] ?? 10000);
    $features = $data['features'] ?? [];
    
    // Validation
    if (!validate_license_type($license_type)) {
        return ['error' => 'Invalid license type'];
    }
    
    if (!empty($hardware_id) && !validate_hardware_id_format($hardware_id)) {
        return ['error' => 'Invalid hardware ID format'];
    }
    
    if (!empty($customer_email) && !validate_email($customer_email)) {
        return ['error' => 'Invalid email format'];
    }
    
    if ($expires_days < 1 || $expires_days > 3650) {
        return ['error' => 'Invalid expiration days (1-3650)'];
    }
    
    if ($max_threads < 1 || $max_threads > 999999) {
        return ['error' => 'Invalid max threads (1-999999)'];
    }
    
    if ($max_recipients < 1 || $max_recipients > 999999999) {
        return ['error' => 'Invalid max recipients (1-999999999)'];
    }
    
    // Generate license key
    $license_data = [
        'type' => $license_type,
        'hardware_id' => $hardware_id,
        'expires' => date('Y-m-d H:i:s', strtotime("+{$expires_days} days")),
        'features' => $features,
        'max_threads' => $max_threads,
        'max_recipients' => $max_recipients,
        'generated' => date('Y-m-d H:i:s')
    ];
    
    $license_key = generate_license_key($license_data);
    
    // Save to database
    $db_data = [
        'license_key' => $license_key,
        'license_type' => $license_type,
        'hardware_id' => $hardware_id,
        'customer_name' => $customer_name,
        'customer_email' => $customer_email,
        'expires' => $license_data['expires'],
        'max_threads' => $max_threads,
        'max_recipients' => $max_recipients,
        'features' => $features
    ];
    
    $license_id = create_license_db($db_data);
    
    if ($license_id) {
        log_message("License created: ID $license_id, Type: $license_type");
        return [
            'success' => true,
            'license_id' => $license_id,
            'license_key' => $license_key,
            'message' => 'License created successfully'
        ];
    } else {
        return ['error' => 'Failed to create license'];
    }
}

/**
 * Update license
 */
function update_license($id, $data) {
    $license = get_license_by_id($id);
    
    if (!$license) {
        return ['error' => 'License not found'];
    }
    
    $update_data = [];
    
    if (isset($data['license_type'])) {
        $type = sanitize_input($data['license_type'], 50);
        if (!validate_license_type($type)) {
            return ['error' => 'Invalid license type'];
        }
        $update_data['license_type'] = $type;
    }
    
    if (isset($data['hardware_id'])) {
        $hw_id = sanitize_input($data['hardware_id'], 100);
        if (!empty($hw_id) && !validate_hardware_id_format($hw_id)) {
            return ['error' => 'Invalid hardware ID format'];
        }
        $update_data['hardware_id'] = $hw_id;
    }
    
    if (isset($data['customer_name'])) {
        $update_data['customer_name'] = sanitize_input($data['customer_name'], 200);
    }
    
    if (isset($data['customer_email'])) {
        $email = sanitize_input($data['customer_email'], 200);
        if (!empty($email) && !validate_email($email)) {
            return ['error' => 'Invalid email format'];
        }
        $update_data['customer_email'] = $email;
    }
    
    if (isset($data['expires_days'])) {
        $days = (int)$data['expires_days'];
        if ($days >= 1 && $days <= 3650) {
            $update_data['expires'] = date('Y-m-d H:i:s', strtotime("+{$days} days"));
        }
    }
    
    if (isset($data['max_threads'])) {
        $threads = (int)$data['max_threads'];
        if ($threads >= 1 && $threads <= 999999) {
            $update_data['max_threads'] = $threads;
        }
    }
    
    if (isset($data['max_recipients'])) {
        $recipients = (int)$data['max_recipients'];
        if ($recipients >= 1 && $recipients <= 999999999) {
            $update_data['max_recipients'] = $recipients;
        }
    }
    
    if (isset($data['features'])) {
        $update_data['features'] = $data['features'];
    }
    
    if (isset($data['is_active'])) {
        $update_data['is_active'] = (bool)$data['is_active'];
    }
    
    if (empty($update_data)) {
        return ['error' => 'No fields to update'];
    }
    
    $result = update_license_db($id, $update_data);
    
    if ($result) {
        log_message("License updated: ID $id");
        return ['success' => true, 'message' => 'License updated successfully'];
    } else {
        return ['error' => 'Failed to update license'];
    }
}

/**
 * Delete license
 */
function delete_license($id) {
    $license = get_license_by_id($id);
    
    if (!$license) {
        return ['error' => 'License not found'];
    }
    
    $result = delete_license_db($id);
    
    if ($result) {
        log_message("License deactivated: ID $id");
        return ['success' => true, 'message' => 'License deactivated successfully'];
    } else {
        return ['error' => 'Failed to deactivate license'];
    }
}

/**
 * Check health
 */
function check_health() {
    $db_healthy = check_db_health();
    
    return [
        'status' => $db_healthy ? 'healthy' : 'unhealthy',
        'database' => $db_healthy ? 'connected' : 'disconnected',
        'timestamp' => time()
    ];
}
