<?php
/**
 * Telegram Bot for License Management
 * PHP version - Works with webhook or polling
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/utils.php';

// Load bot configuration
$bot_config_file = __DIR__ . '/bot_config.json';

if (!file_exists($bot_config_file)) {
    die("Error: bot_config.json not found. Please create it with your bot token and admin user IDs.\n");
}

$bot_config = json_decode(file_get_contents($bot_config_file), true);

if (!$bot_config || !isset($bot_config['bot_token']) || !isset($bot_config['admin_user_ids'])) {
    die("Error: Invalid bot_config.json. Required: bot_token, admin_user_ids\n");
}

$BOT_TOKEN = $bot_config['bot_token'];
$ADMIN_USER_IDS = $bot_config['admin_user_ids'];
$API_URL = "https://api.telegram.org/bot{$BOT_TOKEN}";

/**
 * Send message to Telegram
 */
function sendMessage($chat_id, $text, $reply_markup = null) {
    global $API_URL;
    
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    
    $ch = curl_init($API_URL . '/sendMessage');
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $result = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($result, true);
}

/**
 * Check if user is admin
 */
function isAdmin($user_id) {
    global $ADMIN_USER_IDS;
    return in_array($user_id, $ADMIN_USER_IDS);
}

/**
 * Get main menu keyboard
 */
function getMainMenuKeyboard() {
    return [
        'inline_keyboard' => [
            [
                ['text' => '➕ Add License', 'callback_data' => 'menu_add'],
                ['text' => '📋 List Licenses', 'callback_data' => 'menu_list']
            ],
            [
                ['text' => '✏️ Edit License', 'callback_data' => 'menu_edit'],
                ['text' => '🗑️ Delete License', 'callback_data' => 'menu_delete']
            ],
            [
                ['text' => 'ℹ️ License Info', 'callback_data' => 'menu_info'],
                ['text' => '❓ Help', 'callback_data' => 'menu_help']
            ]
        ]
    ];
}

/**
 * Get license type keyboard
 */
function getLicenseTypeKeyboard() {
    return [
        'inline_keyboard' => [
            [
                ['text' => 'Basic', 'callback_data' => 'type_basic'],
                ['text' => 'Premium', 'callback_data' => 'type_premium']
            ],
            [
                ['text' => 'Enterprise', 'callback_data' => 'type_enterprise'],
                ['text' => 'Trial', 'callback_data' => 'type_trial']
            ],
            [
                ['text' => '🔙 Back', 'callback_data' => 'menu_main']
            ]
        ]
    ];
}

/**
 * Handle /start command
 */
function handleStart($chat_id, $user_id) {
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ <b>Access Denied</b>\n\nYou are not authorized to use this bot.");
        return;
    }
    
    $text = "🔐 <b>PADRINO License Bot</b>\n\n";
    $text .= "Welcome to the license management bot!\n\n";
    $text .= "Use the buttons below to manage licenses:";
    
    sendMessage($chat_id, $text, getMainMenuKeyboard());
}

/**
 * Handle /help command
 */
function handleHelp($chat_id) {
    $text = "📖 <b>Bot Commands</b>\n\n";
    $text .= "/start - Show main menu\n";
    $text .= "/help - Show this help\n";
    $text .= "/add - Add new license\n";
    $text .= "/list - List all licenses\n";
    $text .= "/info <license_id> - Get license info\n";
    $text .= "/delete <license_id> - Delete license\n\n";
    $text .= "Or use the buttons in the menu!";
    
    sendMessage($chat_id, $text);
}

/**
 * Handle /add command
 */
function handleAdd($chat_id) {
    $text = "➕ <b>Add New License</b>\n\n";
    $text .= "Select license type:";
    
    sendMessage($chat_id, $text, getLicenseTypeKeyboard());
}

/**
 * Handle /list command
 */
function handleList($chat_id) {
    $licenses = get_all_licenses();
    
    if (empty($licenses)) {
        sendMessage($chat_id, "📋 <b>No Licenses Found</b>\n\nThere are no licenses in the database.");
        return;
    }
    
    $text = "📋 <b>All Licenses</b>\n\n";
    $text .= "Total: " . count($licenses) . "\n\n";
    
    foreach (array_slice($licenses, 0, 10) as $license) {
        $expires = new DateTime($license['expires']);
        $isExpired = $expires < new DateTime();
        $status = $license['is_active'] && !$isExpired ? '✅ Active' : '❌ Inactive';
        
        $text .= "<b>#{$license['id']}</b> - {$license['license_type']}\n";
        $text .= "Status: {$status}\n";
        $text .= "Expires: " . $expires->format('Y-m-d') . "\n";
        $text .= "Customer: " . ($license['customer_name'] ?: 'N/A') . "\n\n";
    }
    
    if (count($licenses) > 10) {
        $text .= "... and " . (count($licenses) - 10) . " more";
    }
    
    sendMessage($chat_id, $text, getMainMenuKeyboard());
}

/**
 * Handle /info command
 */
function handleInfo($chat_id, $license_id) {
    $license = get_license_by_id($license_id);
    
    if (!$license) {
        sendMessage($chat_id, "❌ License #{$license_id} not found.");
        return;
    }
    
    $expires = new DateTime($license['expires']);
    $isExpired = $expires < new DateTime();
    $status = $license['is_active'] && !$isExpired ? '✅ Active' : '❌ Inactive';
    
    $text = "ℹ️ <b>License Information</b>\n\n";
    $text .= "<b>ID:</b> #{$license['id']}\n";
    $text .= "<b>Type:</b> {$license['license_type']}\n";
    $text .= "<b>Status:</b> {$status}\n";
    $text .= "<b>Expires:</b> " . $expires->format('Y-m-d H:i:s') . "\n";
    $text .= "<b>Max Threads:</b> {$license['max_threads']}\n";
    $text .= "<b>Max Recipients:</b> {$license['max_recipients']}\n";
    $text .= "<b>Hardware ID:</b> " . ($license['hardware_id'] ?: 'Not bound') . "\n";
    $text .= "<b>Customer:</b> " . ($license['customer_name'] ?: 'N/A') . "\n";
    $text .= "<b>Email:</b> " . ($license['customer_email'] ?: 'N/A') . "\n";
    $text .= "<b>Created:</b> " . (new DateTime($license['created_at']))->format('Y-m-d H:i:s') . "\n";
    
    $text .= "\n<b>License Key:</b>\n<code>" . substr($license['license_key'], 0, 50) . "...</code>";
    
    sendMessage($chat_id, $text, getMainMenuKeyboard());
}

/**
 * Handle /delete command
 */
function handleDelete($chat_id, $license_id) {
    $license = get_license_by_id($license_id);
    
    if (!$license) {
        sendMessage($chat_id, "❌ License #{$license_id} not found.");
        return;
    }
    
    $result = delete_license($license_id);
    
    if (isset($result['success']) && $result['success']) {
        sendMessage($chat_id, "✅ License #{$license_id} deactivated successfully.", getMainMenuKeyboard());
    } else {
        sendMessage($chat_id, "❌ Failed to deactivate license: " . ($result['error'] ?? 'Unknown error'));
    }
}

/**
 * Handle callback queries
 */
function handleCallback($callback_query) {
    $chat_id = $callback_query['message']['chat']['id'];
    $user_id = $callback_query['from']['id'];
    $data = $callback_query['data'];
    
    if (!isAdmin($user_id)) {
        sendMessage($chat_id, "❌ Access Denied");
        return;
    }
    
    // Answer callback query
    global $API_URL;
    $ch = curl_init($API_URL . '/answerCallbackQuery');
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
        'callback_query_id' => $callback_query['id']
    ]));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_exec($ch);
    curl_close($ch);
    
    // Handle menu actions
    if ($data === 'menu_main') {
        handleStart($chat_id, $user_id);
    } elseif ($data === 'menu_add') {
        handleAdd($chat_id);
    } elseif ($data === 'menu_list') {
        handleList($chat_id);
    } elseif ($data === 'menu_info') {
        sendMessage($chat_id, "ℹ️ <b>License Info</b>\n\nSend: /info <license_id>\n\nExample: /info 1");
    } elseif ($data === 'menu_delete') {
        sendMessage($chat_id, "🗑️ <b>Delete License</b>\n\nSend: /delete <license_id>\n\nExample: /delete 1");
    } elseif ($data === 'menu_help') {
        handleHelp($chat_id);
    } elseif (strpos($data, 'type_') === 0) {
        $type = str_replace('type_', '', $data);
        sendMessage($chat_id, "✅ License type selected: <b>{$type}</b>\n\nTo create a license, use the web panel at:\nhttps://your-domain.com/license/admin.php\n\nOr use /add command with parameters.");
    }
}

/**
 * Process webhook update
 */
function processUpdate($update) {
    if (isset($update['message'])) {
        $message = $update['message'];
        $chat_id = $message['chat']['id'];
        $user_id = $message['from']['id'];
        $text = $message['text'] ?? '';
        
        if (strpos($text, '/start') === 0) {
            handleStart($chat_id, $user_id);
        } elseif (strpos($text, '/help') === 0) {
            handleHelp($chat_id);
        } elseif (strpos($text, '/add') === 0) {
            handleAdd($chat_id);
        } elseif (strpos($text, '/list') === 0) {
            handleList($chat_id);
        } elseif (strpos($text, '/info') === 0) {
            $parts = explode(' ', $text);
            $license_id = isset($parts[1]) ? (int)$parts[1] : 0;
            if ($license_id) {
                handleInfo($chat_id, $license_id);
            } else {
                sendMessage($chat_id, "Usage: /info <license_id>");
            }
        } elseif (strpos($text, '/delete') === 0) {
            $parts = explode(' ', $text);
            $license_id = isset($parts[1]) ? (int)$parts[1] : 0;
            if ($license_id) {
                handleDelete($chat_id, $license_id);
            } else {
                sendMessage($chat_id, "Usage: /delete <license_id>");
            }
        }
    } elseif (isset($update['callback_query'])) {
        handleCallback($update['callback_query']);
    }
}

// Webhook mode (for cPanel)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $update = json_decode(file_get_contents('php://input'), true);
    if ($update) {
        processUpdate($update);
        http_response_code(200);
        echo "OK";
    }
} else {
    // Polling mode (for testing)
    echo "Telegram Bot is running in polling mode.\n";
    echo "Set up webhook for production use.\n\n";
    
    // Get updates
    $ch = curl_init($API_URL . '/getUpdates');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $result = curl_exec($ch);
    curl_close($ch);
    
    $updates = json_decode($result, true);
    
    if (isset($updates['result']) && is_array($updates['result'])) {
        foreach ($updates['result'] as $update) {
            processUpdate($update);
        }
    }
}
