<?php
/**
 * PADRINO License Server - Main Entry Point
 * PHP version for cPanel hosting (no setup required!)
 */

// Start session
session_start();

// Set headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Load configuration and utilities
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/utils.php';

// Initialize database
init_database();

// Get request path
$request_uri = $_SERVER['REQUEST_URI'];
$script_name = $_SERVER['SCRIPT_NAME'];
$path = str_replace(dirname($script_name), '', $request_uri);
$path = trim($path, '/');
$path_parts = explode('/', $path);

// Remove empty parts
$path_parts = array_filter($path_parts, function($part) {
    return !empty($part);
});
$path_parts = array_values($path_parts);

// Route handling
$method = $_SERVER['REQUEST_METHOD'];
$route = isset($path_parts[0]) ? $path_parts[0] : '';
$sub_route = isset($path_parts[1]) ? $path_parts[1] : '';

// Serve pages
if ($route === '' || $route === 'index.php') {
    // Root - redirect to login or dashboard
    if (is_authenticated()) {
        header('Location: /license/admin.php');
        exit;
    } else {
        header('Location: /license/login.php');
        exit;
    }
}

// API routes
if ($route === 'api') {
    handle_api($method, $sub_route, $path_parts);
    exit;
}

// Page routes
if ($route === 'login.php' || $route === 'login') {
    require_once __DIR__ . '/login.php';
    exit;
}

if ($route === 'admin.php' || $route === 'admin') {
    if (!is_authenticated()) {
        header('Location: /license/login.php');
        exit;
    }
    require_once __DIR__ . '/admin.php';
    exit;
}

// 404
http_response_code(404);
echo json_encode(['error' => 'Not found']);
exit;

/**
 * Handle API requests
 */
function handle_api($method, $sub_route, $path_parts) {
    global $db_file;
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    if ($input === null) {
        $input = [];
    }
    
    // Health check (public)
    if ($sub_route === 'health') {
        $status = check_health();
        echo json_encode($status);
        exit;
    }
    
    // Validate license (public)
    if ($sub_route === 'validate' && $method === 'POST') {
        $license_key = $input['license_key'] ?? '';
        $hardware_id = $input['hardware_id'] ?? '';
        
        $result = validate_license($license_key, $hardware_id);
        echo json_encode($result);
        exit;
    }
    
    // Validate hardware ID format (public)
    if ($sub_route === 'hardware' && $path_parts[2] === 'validate' && $method === 'POST') {
        $hardware_id = $input['hardware_id'] ?? '';
        $valid = validate_hardware_id_format($hardware_id);
        echo json_encode(['valid' => $valid]);
        exit;
    }
    
    // Login (public)
    if ($sub_route === 'login' && $method === 'POST') {
        $username = $input['username'] ?? '';
        $password = $input['password'] ?? '';
        
        if (authenticate($username, $password)) {
            $_SESSION['authenticated'] = true;
            $_SESSION['username'] = $username;
            echo json_encode(['success' => true, 'message' => 'Login successful']);
        } else {
            http_response_code(401);
            echo json_encode(['error' => 'Invalid credentials']);
        }
        exit;
    }
    
    // Logout (requires auth)
    if ($sub_route === 'logout' && $method === 'POST') {
        if (!is_authenticated()) {
            http_response_code(401);
            echo json_encode(['error' => 'Unauthorized']);
            exit;
        }
        session_destroy();
        echo json_encode(['success' => true, 'message' => 'Logged out']);
        exit;
    }
    
    // Auth status (requires auth)
    if ($sub_route === 'auth' && $path_parts[2] === 'status') {
        echo json_encode([
            'authenticated' => is_authenticated(),
            'username' => $_SESSION['username'] ?? null
        ]);
        exit;
    }
    
    // All other API routes require authentication
    if (!is_authenticated()) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }
    
    // Licenses endpoints
    if ($sub_route === 'licenses') {
        if ($method === 'GET') {
            // Get all licenses
            $licenses = get_all_licenses();
            echo json_encode(['licenses' => $licenses]);
            exit;
        }
        
        if ($method === 'POST') {
            // Create license
            $result = create_license($input);
            echo json_encode($result);
            exit;
        }
        
        // PUT /api/licenses/<id>
        if ($method === 'PUT' && isset($path_parts[2])) {
            $id = (int)$path_parts[2];
            $result = update_license($id, $input);
            echo json_encode($result);
            exit;
        }
        
        // DELETE /api/licenses/<id>
        if ($method === 'DELETE' && isset($path_parts[2])) {
            $id = (int)$path_parts[2];
            $result = delete_license($id);
            echo json_encode($result);
            exit;
        }
    }
    
    // 404
    http_response_code(404);
    echo json_encode(['error' => 'Not found']);
    exit;
}
