<?php
/**
 * Database utilities for License Server
 */

require_once __DIR__ . '/config.php';

/**
 * Get database connection
 */
function get_db() {
    $db = new SQLite3(DB_FILE);
    $db->busyTimeout(10000); // 10 seconds
    $db->exec('PRAGMA journal_mode=WAL');
    $db->exec('PRAGMA synchronous=NORMAL');
    $db->exec('PRAGMA foreign_keys=ON');
    return $db;
}

/**
 * Initialize database schema
 */
function init_database() {
    $db = get_db();
    
    $db->exec('
        CREATE TABLE IF NOT EXISTS licenses (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            license_key TEXT UNIQUE NOT NULL,
            license_type TEXT NOT NULL,
            hardware_id TEXT,
            customer_name TEXT,
            customer_email TEXT,
            expires TEXT,
            max_threads INTEGER DEFAULT 10,
            max_recipients INTEGER DEFAULT 10000,
            features TEXT,
            created_at TEXT DEFAULT CURRENT_TIMESTAMP,
            activated_at TEXT,
            is_active INTEGER DEFAULT 1
        )
    ');
    
    $db->exec('
        CREATE INDEX IF NOT EXISTS idx_license_key ON licenses(license_key)
    ');
    
    $db->exec('
        CREATE INDEX IF NOT EXISTS idx_hardware_id ON licenses(hardware_id)
    ');
    
    $db->exec('
        CREATE INDEX IF NOT EXISTS idx_is_active ON licenses(is_active)
    ');
    
    $db->close();
    
    log_message('Database initialized');
}

/**
 * Get all licenses
 */
function get_all_licenses() {
    $db = get_db();
    $stmt = $db->prepare('SELECT * FROM licenses ORDER BY created_at DESC');
    $result = $stmt->execute();
    
    $licenses = [];
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
        $row['features'] = json_decode($row['features'] ?? '{}', true);
        $row['is_active'] = (bool)$row['is_active'];
        $licenses[] = $row;
    }
    
    $db->close();
    return $licenses;
}

/**
 * Get license by key
 */
function get_license_by_key($license_key) {
    $db = get_db();
    $stmt = $db->prepare('SELECT * FROM licenses WHERE license_key = ?');
    $stmt->bindValue(1, $license_key, SQLITE3_TEXT);
    $result = $stmt->execute();
    
    $row = $result->fetchArray(SQLITE3_ASSOC);
    $db->close();
    
    if ($row) {
        $row['features'] = json_decode($row['features'] ?? '{}', true);
        $row['is_active'] = (bool)$row['is_active'];
    }
    
    return $row ?: null;
}

/**
 * Get license by ID
 */
function get_license_by_id($id) {
    $db = get_db();
    $stmt = $db->prepare('SELECT * FROM licenses WHERE id = ?');
    $stmt->bindValue(1, $id, SQLITE3_INTEGER);
    $result = $stmt->execute();
    
    $row = $result->fetchArray(SQLITE3_ASSOC);
    $db->close();
    
    if ($row) {
        $row['features'] = json_decode($row['features'] ?? '{}', true);
        $row['is_active'] = (bool)$row['is_active'];
    }
    
    return $row ?: null;
}

/**
 * Create new license
 */
function create_license_db($data) {
    $db = get_db();
    
    $stmt = $db->prepare('
        INSERT INTO licenses 
        (license_key, license_type, hardware_id, customer_name, customer_email,
         expires, max_threads, max_recipients, features, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ');
    
    $stmt->bindValue(1, $data['license_key'], SQLITE3_TEXT);
    $stmt->bindValue(2, $data['license_type'], SQLITE3_TEXT);
    $stmt->bindValue(3, $data['hardware_id'] ?? null, SQLITE3_TEXT);
    $stmt->bindValue(4, $data['customer_name'] ?? null, SQLITE3_TEXT);
    $stmt->bindValue(5, $data['customer_email'] ?? null, SQLITE3_TEXT);
    $stmt->bindValue(6, $data['expires'], SQLITE3_TEXT);
    $stmt->bindValue(7, $data['max_threads'], SQLITE3_INTEGER);
    $stmt->bindValue(8, $data['max_recipients'], SQLITE3_INTEGER);
    $stmt->bindValue(9, json_encode($data['features'] ?? []), SQLITE3_TEXT);
    $stmt->bindValue(10, date('Y-m-d H:i:s'), SQLITE3_TEXT);
    
    $result = $stmt->execute();
    $license_id = $db->lastInsertRowID();
    $db->close();
    
    return $license_id;
}

/**
 * Update license
 */
function update_license_db($id, $data) {
    $db = get_db();
    
    $fields = [];
    $values = [];
    
    if (isset($data['license_type'])) {
        $fields[] = 'license_type = ?';
        $values[] = $data['license_type'];
    }
    if (isset($data['hardware_id'])) {
        $fields[] = 'hardware_id = ?';
        $values[] = $data['hardware_id'];
    }
    if (isset($data['customer_name'])) {
        $fields[] = 'customer_name = ?';
        $values[] = $data['customer_name'];
    }
    if (isset($data['customer_email'])) {
        $fields[] = 'customer_email = ?';
        $values[] = $data['customer_email'];
    }
    if (isset($data['expires'])) {
        $fields[] = 'expires = ?';
        $values[] = $data['expires'];
    }
    if (isset($data['max_threads'])) {
        $fields[] = 'max_threads = ?';
        $values[] = $data['max_threads'];
    }
    if (isset($data['max_recipients'])) {
        $fields[] = 'max_recipients = ?';
        $values[] = $data['max_recipients'];
    }
    if (isset($data['features'])) {
        $fields[] = 'features = ?';
        $values[] = json_encode($data['features']);
    }
    if (isset($data['is_active'])) {
        $fields[] = 'is_active = ?';
        $values[] = $data['is_active'] ? 1 : 0;
    }
    
    if (empty($fields)) {
        $db->close();
        return false;
    }
    
    $sql = 'UPDATE licenses SET ' . implode(', ', $fields) . ' WHERE id = ?';
    $values[] = $id;
    
    $stmt = $db->prepare($sql);
    for ($i = 0; $i < count($values); $i++) {
        $stmt->bindValue($i + 1, $values[$i]);
    }
    
    $result = $stmt->execute();
    $db->close();
    
    return $result !== false;
}

/**
 * Delete/deactivate license
 */
function delete_license_db($id) {
    $db = get_db();
    $stmt = $db->prepare('UPDATE licenses SET is_active = 0 WHERE id = ?');
    $stmt->bindValue(1, $id, SQLITE3_INTEGER);
    $result = $stmt->execute();
    $db->close();
    
    return $result !== false;
}

/**
 * Check database health
 */
function check_db_health() {
    try {
        $db = get_db();
        $result = $db->query('SELECT 1');
        $db->close();
        return $result !== false;
    } catch (Exception $e) {
        log_message('Database health check failed: ' . $e->getMessage());
        return false;
    }
}
