<?php
/**
 * Admin Dashboard - Premium UI Design
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db.php';

// Check authentication
if (!is_authenticated()) {
    header('Location: login.php');
    exit;
}

// Get stats
$db = get_db();
$total_licenses = $db->querySingle('SELECT COUNT(*) FROM licenses');
$active_licenses = $db->querySingle('SELECT COUNT(*) FROM licenses WHERE is_active = 1');
$expired_licenses = $db->querySingle('SELECT COUNT(*) FROM licenses WHERE expires < datetime("now")');
$db->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PADRINO License Server - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --danger: #ef4444;
            --warning: #f59e0b;
            --info: #3b82f6;
            --dark: #1a1a1a;
            --light: #f8f9fa;
            --border: #e5e7eb;
        }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            color: #1a1a1a;
        }
        
        .container {
            max-width: 1600px;
            margin: 0 auto;
        }
        
        /* Header */
        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 24px 32px;
            border-radius: 16px;
            margin-bottom: 24px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
            border: 1px solid rgba(255,255,255,0.2);
        }
        
        .header-left {
            display: flex;
            align-items: center;
            gap: 16px;
        }
        
        .logo-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        
        .header h1 {
            color: var(--dark);
            font-size: 24px;
            font-weight: 700;
        }
        
        .header p {
            color: #666;
            font-size: 14px;
            margin-top: 2px;
        }
        
        .btn-logout {
            padding: 10px 20px;
            background: var(--danger);
            color: white;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-logout:hover {
            background: #dc2626;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(239, 68, 68, 0.4);
        }
        
        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 24px;
        }
        
        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 24px;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            border: 1px solid rgba(255,255,255,0.2);
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-4px);
        }
        
        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
        }
        
        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            color: white;
        }
        
        .stat-icon.total { background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%); }
        .stat-icon.active { background: linear-gradient(135deg, var(--success) 0%, #059669 100%); }
        .stat-icon.expired { background: linear-gradient(135deg, var(--danger) 0%, #dc2626 100%); }
        .stat-icon.inactive { background: linear-gradient(135deg, var(--warning) 0%, #d97706 100%); }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 4px;
        }
        
        .stat-label {
            font-size: 14px;
            color: #666;
            font-weight: 500;
        }
        
        /* Form Panel */
        .form-panel {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 32px;
            border-radius: 16px;
            margin-bottom: 24px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            border: 1px solid rgba(255,255,255,0.2);
        }
        
        .form-panel h2 {
            color: var(--dark);
            font-size: 20px;
            font-weight: 700;
            margin-bottom: 24px;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .form-panel h2 i {
            color: var(--primary);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--dark);
            font-size: 14px;
        }
        
        .form-group input,
        .form-group select {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid var(--border);
            border-radius: 10px;
            font-size: 14px;
            transition: all 0.3s ease;
            background: white;
        }
        
        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1);
        }
        
        .form-group small {
            display: block;
            margin-top: 6px;
            color: #666;
            font-size: 12px;
        }
        
        .btn {
            padding: 14px 28px;
            border: none;
            border-radius: 10px;
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.5);
        }
        
        .btn-danger {
            background: linear-gradient(135deg, var(--danger) 0%, #dc2626 100%);
            box-shadow: 0 4px 15px rgba(239, 68, 68, 0.4);
        }
        
        .btn-danger:hover {
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.5);
        }
        
        /* Licenses Table */
        .licenses-table {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            border: 1px solid rgba(255,255,255,0.2);
        }
        
        .table-header {
            padding: 24px 32px;
            border-bottom: 2px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .table-header h2 {
            color: var(--dark);
            font-size: 20px;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .table-header h2 i {
            color: var(--primary);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th {
            padding: 16px 32px;
            text-align: left;
            background: var(--light);
            font-weight: 600;
            color: var(--dark);
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        td {
            padding: 16px 32px;
            border-top: 1px solid var(--border);
            font-size: 14px;
        }
        
        tr:hover {
            background: var(--light);
        }
        
        .badge {
            padding: 6px 12px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            display: inline-block;
        }
        
        .badge-active {
            background: #d1fae5;
            color: #065f46;
        }
        
        .badge-inactive {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .badge-type {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
            text-transform: uppercase;
            font-size: 11px;
        }
        
        .license-key {
            font-family: 'Courier New', monospace;
            font-size: 11px;
            color: #666;
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .message {
            padding: 16px 20px;
            border-radius: 12px;
            margin-bottom: 24px;
            display: none;
            font-weight: 500;
            animation: slideDown 0.3s ease;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .message.show {
            display: block;
        }
        
        .message-success {
            background: #d1fae5;
            color: #065f46;
            border: 1px solid #a7f3d0;
        }
        
        .message-error {
            background: #fee2e2;
            color: #991b1b;
            border: 1px solid #fecaca;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        
        .empty-state i {
            font-size: 48px;
            color: #ccc;
            margin-bottom: 16px;
        }
        
        .loading {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        
        .loading i {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- Header -->
        <div class="header">
            <div class="header-left">
                <div class="logo-icon">🔐</div>
                <div>
                    <h1>PADRINO License Server</h1>
                    <p>Admin Control Panel</p>
                </div>
            </div>
            <button class="btn-logout" onclick="logout()">
                <i class="fas fa-sign-out-alt"></i> Logout
            </button>
        </div>
        
        <!-- Stats -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value" id="statTotal"><?php echo $total_licenses; ?></div>
                        <div class="stat-label">Total Licenses</div>
                    </div>
                    <div class="stat-icon total">
                        <i class="fas fa-key"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value" id="statActive"><?php echo $active_licenses; ?></div>
                        <div class="stat-label">Active Licenses</div>
                    </div>
                    <div class="stat-icon active">
                        <i class="fas fa-check-circle"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value" id="statExpired"><?php echo $expired_licenses; ?></div>
                        <div class="stat-label">Expired Licenses</div>
                    </div>
                    <div class="stat-icon expired">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <div class="stat-value" id="statInactive"><?php echo $total_licenses - $active_licenses; ?></div>
                        <div class="stat-label">Inactive Licenses</div>
                    </div>
                    <div class="stat-icon inactive">
                        <i class="fas fa-ban"></i>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Message -->
        <div class="message" id="message"></div>
        
        <!-- Create License Form -->
        <div class="form-panel">
            <h2><i class="fas fa-plus-circle"></i> Create New License</h2>
            <form id="createLicenseForm">
                <div class="form-row">
                    <div class="form-group">
                        <label><i class="fas fa-tag"></i> License Type</label>
                        <select id="licenseType" required>
                            <option value="basic">Basic</option>
                            <option value="premium">Premium</option>
                            <option value="enterprise">Enterprise</option>
                            <option value="trial">Trial</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label><i class="fas fa-microchip"></i> Hardware ID (Optional)</label>
                        <input type="text" id="hardwareId" maxlength="64" pattern="[0-9a-fA-F]{64}" placeholder="64-character hex string">
                        <small>Leave empty to bind on first activation</small>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label><i class="fas fa-user"></i> Customer Name</label>
                        <input type="text" id="customerName" placeholder="John Doe">
                    </div>
                    
                    <div class="form-group">
                        <label><i class="fas fa-envelope"></i> Customer Email</label>
                        <input type="email" id="customerEmail" placeholder="customer@example.com">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label><i class="fas fa-calendar-alt"></i> Expires (Days)</label>
                        <input type="number" id="expiresDays" value="365" min="1" max="3650" required>
                    </div>
                    
                    <div class="form-group">
                        <label><i class="fas fa-layer-group"></i> Max Threads</label>
                        <input type="number" id="maxThreads" value="10" min="1" max="999999" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label><i class="fas fa-users"></i> Max Recipients</label>
                    <input type="number" id="maxRecipients" value="10000" min="1" max="999999999" required>
                </div>
                
                <button type="submit" class="btn">
                    <i class="fas fa-plus"></i> Create License
                </button>
            </form>
        </div>
        
        <!-- Licenses Table -->
        <div class="licenses-table">
            <div class="table-header">
                <h2><i class="fas fa-list"></i> All Licenses</h2>
            </div>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Type</th>
                        <th>License Key</th>
                        <th>Hardware ID</th>
                        <th>Customer</th>
                        <th>Expires</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="licensesTableBody">
                    <tr>
                        <td colspan="8" class="loading">
                            <i class="fas fa-spinner"></i> Loading licenses...
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    
    <script>
        const baseUrl = window.location.pathname.replace(/\/[^\/]*$/, '');
        
        // Load licenses on page load
        loadLicenses();
        
        // Create license form
        document.getElementById('createLicenseForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const btn = e.target.querySelector('button[type="submit"]');
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creating...';
            
            const data = {
                license_type: document.getElementById('licenseType').value,
                hardware_id: document.getElementById('hardwareId').value,
                customer_name: document.getElementById('customerName').value,
                customer_email: document.getElementById('customerEmail').value,
                expires_days: parseInt(document.getElementById('expiresDays').value),
                max_threads: parseInt(document.getElementById('maxThreads').value),
                max_recipients: parseInt(document.getElementById('maxRecipients').value)
            };
            
            try {
                const res = await fetch(baseUrl + '/api/licenses', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(data)
                });
                
                const result = await res.json();
                
                if (res.ok && result.success) {
                    showMessage('✅ License created successfully! Key: ' + result.license_key.substring(0, 50) + '...', 'success');
                    document.getElementById('createLicenseForm').reset();
                    loadLicenses();
                    updateStats();
                } else {
                    showMessage('❌ ' + (result.error || 'Failed to create license'), 'error');
                }
            } catch (e) {
                showMessage('❌ Error: ' + e.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        });
        
        // Load licenses
        async function loadLicenses() {
            try {
                const res = await fetch(baseUrl + '/api/licenses');
                const data = await res.json();
                
                if (res.ok && data.licenses) {
                    renderLicenses(data.licenses);
                } else {
                    document.getElementById('licensesTableBody').innerHTML = 
                        '<tr><td colspan="8" class="empty-state"><i class="fas fa-exclamation-circle"></i><br>Error loading licenses</td></tr>';
                }
            } catch (e) {
                document.getElementById('licensesTableBody').innerHTML = 
                    '<tr><td colspan="8" class="empty-state"><i class="fas fa-exclamation-circle"></i><br>Error: ' + e.message + '</td></tr>';
            }
        }
        
        // Render licenses table
        function renderLicenses(licenses) {
            const tbody = document.getElementById('licensesTableBody');
            
            if (licenses.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" class="empty-state"><i class="fas fa-inbox"></i><br>No licenses found. Create your first license above!</td></tr>';
                return;
            }
            
            tbody.innerHTML = licenses.map(license => {
                const expires = new Date(license.expires);
                const isExpired = expires < new Date();
                const status = license.is_active && !isExpired ? 'active' : 'inactive';
                const statusText = isExpired ? 'Expired' : (license.is_active ? 'Active' : 'Inactive');
                
                return `
                    <tr>
                        <td><strong>#${license.id}</strong></td>
                        <td><span class="badge badge-type">${license.license_type}</span></td>
                        <td class="license-key" title="${license.license_key}">${license.license_key.substring(0, 40)}...</td>
                        <td><small>${license.hardware_id ? license.hardware_id.substring(0, 16) + '...' : '<em style="color:#999">Not bound</em>'}</small></td>
                        <td>
                            ${license.customer_name || '<em style="color:#999">-</em>'}
                            ${license.customer_email ? '<br><small style="color:#666">' + license.customer_email + '</small>' : ''}
                        </td>
                        <td>${expires.toLocaleDateString()}</td>
                        <td><span class="badge badge-${status}">${statusText}</span></td>
                        <td>
                            <button class="btn btn-danger" onclick="deleteLicense(${license.id})" style="padding: 8px 16px; font-size: 13px;">
                                <i class="fas fa-trash"></i> Delete
                            </button>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        
        // Delete license
        async function deleteLicense(id) {
            if (!confirm('Are you sure you want to deactivate this license? This action cannot be undone.')) {
                return;
            }
            
            try {
                const res = await fetch(baseUrl + '/api/licenses/' + id, {
                    method: 'DELETE'
                });
                
                const result = await res.json();
                
                if (res.ok && result.success) {
                    showMessage('✅ License deactivated successfully', 'success');
                    loadLicenses();
                    updateStats();
                } else {
                    showMessage('❌ ' + (result.error || 'Failed to deactivate license'), 'error');
                }
            } catch (e) {
                showMessage('❌ Error: ' + e.message, 'error');
            }
        }
        
        // Update stats
        async function updateStats() {
            try {
                const res = await fetch(baseUrl + '/api/licenses');
                const data = await res.json();
                
                if (res.ok && data.licenses) {
                    const total = data.licenses.length;
                    const active = data.licenses.filter(l => l.is_active && new Date(l.expires) > new Date()).length;
                    const expired = data.licenses.filter(l => new Date(l.expires) < new Date()).length;
                    const inactive = total - active;
                    
                    document.getElementById('statTotal').textContent = total;
                    document.getElementById('statActive').textContent = active;
                    document.getElementById('statExpired').textContent = expired;
                    document.getElementById('statInactive').textContent = inactive;
                }
            } catch (e) {
                console.error('Error updating stats:', e);
            }
        }
        
        // Logout
        async function logout() {
            if (confirm('Are you sure you want to logout?')) {
                try {
                    await fetch(baseUrl + '/api/logout', {method: 'POST'});
                    window.location.href = baseUrl + '/login.php';
                } catch (e) {
                    window.location.href = baseUrl + '/login.php';
                }
            }
        }
        
        // Show message
        function showMessage(text, type) {
            const msgDiv = document.getElementById('message');
            msgDiv.textContent = text;
            msgDiv.className = 'message message-' + type + ' show';
            setTimeout(() => {
                msgDiv.classList.remove('show');
            }, 5000);
        }
    </script>
</body>
</html>
