# How the PHP License Server Works

## Overview

The PHP License Server is a complete license management system that validates licenses for the PADRINO mailer application. It works entirely in PHP with SQLite database - no external dependencies needed!

## Architecture

```
┌─────────────────┐
│   Mailer App   │  (start.py)
│  (Client Side) │
└────────┬────────┘
         │
         │ HTTP POST /api/validate
         │ {license_key, hardware_id}
         ▼
┌─────────────────┐
│  License Server │  (index.php)
│   (PHP Server)  │
└────────┬────────┘
         │
         │ Check database
         ▼
┌─────────────────┐
│  SQLite Database│  (licenses.db)
│   - licenses    │
│   - hardware_id│
│   - expiration  │
└─────────────────┘
```

## How It Works - Step by Step

### 1. User Runs Mailer (start.py)

When user runs `start.py`:
- Generates hardware ID (machine fingerprint)
- Checks for `license.key` file
- If no license, prompts for activation key
- If license exists, validates it

### 2. License Validation Request

The mailer sends HTTP POST to license server:

```http
POST https://your-domain.com/license/api/validate
Content-Type: application/json

{
  "license_key": "eyJ0eXBlIjoiYmFzaWMiLCJleHBpcmVzIjoiMjAyNS0wMS0wMSIs...",
  "hardware_id": "a1b2c3d4e5f6a1b2c3d4e5f6a1b2c3d4e5f6a1b2c3d4e5f6a1b2c3d4e5f6"
}
```

### 3. Server Processing (index.php)

The server:
1. **Receives request** via `index.php` router
2. **Routes to API handler** (`handle_api()` function)
3. **Calls `validate_license()`** in `utils.php`
4. **Checks database** for license key
5. **Validates:**
   - License exists?
   - License is active?
   - License not expired?
   - Hardware ID matches? (if bound)
6. **Returns response**

### 4. Validation Response

**If Valid:**
```json
{
  "valid": true,
  "license_data": {
    "license_type": "premium",
    "expires": "2025-12-31",
    "max_threads": 50,
    "max_recipients": 100000,
    "hardware_id": "a1b2c3d4..."
  },
  "features": {
    "microsoft365": true,
    "proxy_rotation": true
  }
}
```

**If Invalid:**
```json
{
  "valid": false,
  "error": "License is bound to different machine ID"
}
```

### 5. Mailer Response

- **If valid:** Mailer proceeds with campaign
- **If invalid:** Mailer shows error and exits

## Admin Panel Flow

### 1. Admin Login

1. Admin visits: `https://your-domain.com/license/login.php`
2. Enters credentials: `admin` / `admin123`
3. Server validates via `auth.php`
4. Creates session
5. Redirects to `admin.php`

### 2. Create License

1. Admin fills form in `admin.php`
2. Submits via JavaScript to `/api/licenses` (POST)
3. Server processes in `index.php` → `create_license()`
4. Generates license key (base64 encoded JSON)
5. Saves to database via `db.php`
6. Returns license key to admin
7. Admin copies and sends to user

### 3. View Licenses

1. Admin page loads
2. JavaScript fetches `/api/licenses` (GET)
3. Server queries database
4. Returns all licenses
5. JavaScript renders table

## Database Structure

### licenses Table

```sql
CREATE TABLE licenses (
    id INTEGER PRIMARY KEY,
    license_key TEXT UNIQUE,      -- Base64 encoded license data
    license_type TEXT,            -- basic, premium, enterprise
    hardware_id TEXT,             -- 64-char hex (bound machine)
    customer_name TEXT,
    customer_email TEXT,
    expires TEXT,                 -- ISO date string
    max_threads INTEGER,
    max_recipients INTEGER,
    features TEXT,                -- JSON string
    created_at TEXT,
    activated_at TEXT,
    is_active INTEGER             -- 1 = active, 0 = deactivated
)
```

## File Flow

### Request Flow

```
Browser/Client
    ↓
index.php (router)
    ↓
handle_api() function
    ↓
utils.php functions
    ↓
db.php (database)
    ↓
SQLite (licenses.db)
```

### Authentication Flow

```
Login Request
    ↓
index.php → /api/login
    ↓
auth.php → authenticate()
    ↓
admin_users.json (or default)
    ↓
Session created
    ↓
Redirect to admin.php
```

## Security Features

1. **Hardware ID Binding**
   - License tied to specific machine
   - Prevents sharing licenses

2. **Expiration Check**
   - Validates expiration date
   - Blocks expired licenses

3. **Session Authentication**
   - Admin panel requires login
   - Session-based auth

4. **Input Validation**
   - Sanitizes all inputs
   - Validates formats

5. **SQL Injection Protection**
   - Uses prepared statements
   - Parameterized queries

## API Endpoints

### Public (No Auth)

- `POST /api/validate` - Validate license
- `GET /api/health` - Health check

### Admin (Requires Auth)

- `POST /api/login` - Login
- `POST /api/logout` - Logout
- `GET /api/licenses` - List licenses
- `POST /api/licenses` - Create license
- `PUT /api/licenses/<id>` - Update license
- `DELETE /api/licenses/<id>` - Delete license

## License Key Format

License keys are base64-encoded JSON:

```json
{
  "type": "premium",
  "hardware_id": "a1b2c3d4...",
  "expires": "2025-12-31 23:59:59",
  "features": {...},
  "max_threads": 50,
  "max_recipients": 100000,
  "generated": "2024-01-01 12:00:00"
}
```

Encoded as: `eyJ0eXBlIjoicHJlbWl1bSIsImV4cGlyZXMiOiIyMDI1LTEyLTMxIi4uLiJ9`

## Complete Flow Example

### User Activation

1. User runs `python start.py`
2. Sees machine ID: `a1b2c3d4...`
3. Sends machine ID to admin
4. Admin creates license in panel
5. Admin sends license key to user
6. User enters key in `start.py`
7. `start.py` validates with server
8. Server binds hardware ID
9. License saved to `license.key`
10. Mailer starts successfully

### License Validation (Every Run)

1. User runs `python start.py`
2. Reads `license.key` file
3. Sends validation request to server
4. Server checks:
   - License exists? ✅
   - Active? ✅
   - Not expired? ✅
   - Hardware matches? ✅
5. Returns valid response
6. Mailer proceeds

## Error Handling

### Common Errors

1. **License Not Found**
   - Error: "License not found"
   - Solution: Check license key

2. **Hardware Mismatch**
   - Error: "License bound to different machine"
   - Solution: Request new license

3. **License Expired**
   - Error: "License has expired"
   - Solution: Renew license

4. **License Deactivated**
   - Error: "License is deactivated"
   - Solution: Contact admin

## Summary

**Simple Flow:**
1. Admin creates license → Database
2. User activates license → Validates with server
3. Server checks database → Returns validation
4. Mailer uses license → Runs campaign

**All happens via HTTP API calls!**
